//
//  Tile.m
//  15Puzzle
//
//  Created by 久保島 祐磨 on 12/07/19.
//  Copyright 2012年 株式会社 ICT Fractal. All rights reserved.
//

#import "Tile.h"

@interface Tile ()

// 画像を自己と同じサイズに加工して返します
-(UIImage*) shapingImageNamed:(NSString*)imageNamed;

// 通知センタからの通知イベント
-(void) NotifyFromNotificationCenter:(NSNotification*)notification;

// ボタン内の座標中にタッチ座標が含まれているかを返す
-(BOOL) containsTouchLocation:(UITouch*)touch;

// タッチ地点移動によるガイドナンバー表示
-(void) showGuideNumByTouch:(UITouch*)touch;

// スケジュールイベント：画面長押し
-(void) scheduleEventTouchHold:(ccTime)delta;

// 正解フレームのアニメーション表示
-(void) blinkFrame;

@end


@implementation Tile

@dynamic Answer;
-(int) Answer
{
	return _answer;
}
-(void) setAnswer:(int)Answer
{
	_answer = Answer;
	
	if (_lblAnswer == nil) {
		_lblAnswer = [CCLabelTTF labelWithString:@"99" fontName:@"Arial-BoldMT" fontSize:40];
		_lblAnswer.position = CGPointMake(self.contentSize.width / 2, self.contentSize.height / 2);
		_lblAnswer.color = ccc3(255, 100, 40);
		[self addChild:_lblAnswer z:2];
		_lblAnswer.visible = NO;
		
		// 縦横のフォント幅が画像幅の0.8倍に収まるように調整します
		_lblAnswer.scale = MIN((self.contentSize.width * 0.8) / _lblAnswer.contentSize.width, (self.contentSize.height * 0.8) / _lblAnswer.contentSize.height);
		if (_lblAnswer.scale > 1.0) {
			_lblAnswer.scale = 1.0;
		}
	}
	
	[_lblAnswer setString:[NSString stringWithFormat:@"%d", Answer + 1]];
}

@dynamic IsBlank;
-(BOOL) IsBlank
{
	return _isBlank;
}
-(void) setIsBlank:(BOOL)IsBlank
{
	_isBlank = IsBlank;
	
	if (_isBlank) {
		// ブランクタイルの場合、透明度を最大にします
		self.opacity = 0;
	}
	else {
		self.opacity = 255;
	}
}

@synthesize Now = _now;
@synthesize IsTouchHold = _isTouchHold;

-(id) init 
{
	if (self = [super init]) {
		// メンバーの初期化をします
		_imgFrame = nil;
		_imgBlinkFrame = nil;
		_lblAnswer = nil;
		_answer = 0;
		_now = 0;
		_isTouchBegin = NO;
		_isTouchHold = NO;
		_touchLocation = CGPointZero;
		_deltaTime = 0.0;
		_isBlank = NO;
	}
	
	return self;
}

-(void) onEnter
{
	// スーパークラスのonEnterをコールします
	[super onEnter];
	
	// CCTouchDispathcerに登録します（プライオリティ高め）
	[[CCDirector sharedDirector].touchDispatcher addTargetedDelegate:self priority:-9 swallowsTouches:YES];
	
	// 通知センタのオブザーバ登録
	[[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(NotifyFromNotificationCenter:) 
												 name:nil object:nil];
}

-(void) onExit
{
	// スーパークラスのonExitをコールします
	[super onExit];
	
	// CCTouchDispatcherの登録を解除します
	[[CCDirector sharedDirector].touchDispatcher removeDelegate:self];
	
	// 通知センタのオブザーバ登録を解除します
	[[NSNotificationCenter defaultCenter] removeObserver:self];
}

// 画像を自己と同じサイズに加工して返します
-(UIImage*) shapingImageNamed:(NSString*)imageNamed
{
	UIImage* resultImage = [UIImage imageNamed:imageNamed];
	
	UIGraphicsBeginImageContext(CGSizeMake(self.contentSize.width, self.contentSize.height));
	[resultImage drawInRect:CGRectMake(0, 0, self.contentSize.width, self.contentSize.height)];
	resultImage = UIGraphicsGetImageFromCurrentImageContext();
	UIGraphicsEndImageContext();
	
	return resultImage;
}

// フレームを作成する
-(void) createFrame
{
	// 既に枠が作られている場合、解放します
	if (_imgFrame != nil) {
		[_imgFrame removeFromParentAndCleanup:YES];
		_imgFrame = nil;
	}
	if (_imgBlinkFrame != nil) {
		[_imgBlinkFrame removeFromParentAndCleanup:YES];
		_imgBlinkFrame = nil;
	}
	
	// 自分と同じ大きさのフレーム画像を作成します
	_imgFrame = [CCSprite spriteWithCGImage:[self shapingImageNamed:@"Frame.png"].CGImage key:nil];
	_imgFrame.position = CGPointMake(self.contentSize.width / 2, self.contentSize.height / 2);
	_imgFrame.opacity = 127;
	[self addChild:_imgFrame z:1];
	_imgFrame.visible = NO;
	
	_imgBlinkFrame = [CCSprite spriteWithCGImage:[self shapingImageNamed:@"BlinkFrame.png"].CGImage key:nil];
	_imgBlinkFrame.position = CGPointMake(self.contentSize.width / 2, self.contentSize.height / 2);
	_imgBlinkFrame.opacity = 127;
	[self addChild:_imgBlinkFrame z:1];
	_imgBlinkFrame.visible = NO;
}

// タッチ開始
-(BOOL) ccTouchBegan:(UITouch *)touch withEvent:(UIEvent *)event
{	
	BOOL bResult = NO;
	
	// 長押しフラグを初期化します
	_isTouchHold = NO;
	
	if ([self containsTouchLocation:touch]) {
		// 自タイルをタッチされている場合、タッチ座標をcocos2d座標に変換し記録します
		CGPoint touchLocation = [touch locationInView: [touch view]];
		_touchLocation = [[CCDirector sharedDirector] convertToGL:touchLocation];
		
		// 長押し判定のスケジュールを立てます
		_deltaTime = 0.0;
		[self schedule:@selector(scheduleEventTouchHold:)];
		
		// タッチ開始フラグを立てます
		_isTouchBegin = YES;
		bResult = YES;
	}
	
	return bResult;
}

// ボタンタッチ中移動通知
-(void) ccTouchMoved:(UITouch *)touch withEvent:(UIEvent *)event
{	
	// タッチ中の座標をcocos2d座標に変換します
	CGPoint touchLocation = [touch locationInView: [touch view]];
	CGPoint currentTouchLocation = [[CCDirector sharedDirector] convertToGL:touchLocation];
	
	// タッチ開始位置から上下左右方向に一定値以上動いた場合、移動扱いとします
	CGPoint difference = ccpSub(_touchLocation, currentTouchLocation);
	float factor = 20;
	if ((abs(difference.x) > factor) || (abs(difference.y) > factor)) {
		// 移動座標をuserInfoに設定し、移動通知を発行します
		NSDictionary* dic = [NSDictionary dictionaryWithObject:touch forKey:TILE_MSG_NOTIFY_TOUCH_MOVE];
		[[NSNotificationCenter defaultCenter] postNotificationName:TILE_MSG_NOTIFY_TOUCH_MOVE object:self userInfo:dic];
		
		// 基準点を現在のタッチ中座標に変えます
		_touchLocation = currentTouchLocation;
	}
}

// ボタンのタッチ終了通知
-(void) ccTouchEnded:(UITouch *)touch withEvent:(UIEvent *)event
{	
	// 長押しスケジュールを停止します
	[self unschedule:@selector(scheduleEventTouchHold:)];
	
	if (_isTouchBegin == YES) {
		if ([self containsTouchLocation:touch]) {
			if (_isTouchHold == NO) {
				// タップ通知を発行します
				[[NSNotificationCenter defaultCenter] postNotificationName:TILE_MSG_NOTIFY_TAP object:self];
			}
		}
	}
	
	// タッチ終了通知を発行します
	[[NSNotificationCenter defaultCenter] postNotificationName:TILE_MSG_NOTIFY_TOUCH_END object:self];
	
	// タッチフラグを初期化します
	_isTouchBegin = NO;
}

// ボタン内の座標中にタッチ座標が含まれているかを返す
-(BOOL)containsTouchLocation:(UITouch*)touch
{
	// タッチした座標をcocos2d座標に変換します
	CGPoint touchLocation = [touch locationInView: [touch view]];
	CGPoint location = [[CCDirector sharedDirector] convertToGL:touchLocation];
	
	// 親ノードの座標が移動している場合に対応するため、boundingBoxプロパティの値を取得します
	CGRect boundingBox = self.boundingBox;
	
	// CCLayerを継承するクラスまで親ノードをさかのぼって探索します
	CCNode* parent = self.parent;
	while (parent != nil) {
		if ([parent isKindOfClass:[CCLayer class]]) {
			break;
		}
		else {
			parent = parent.parent;
		}
	}
	if (parent != nil) {
		// 親ノードの座標と自ノードの座標を加算します
		boundingBox.origin = ccpAdd(boundingBox.origin, parent.position);
	}
	
	// タッチ座標とボタン内座標の比較結果を返します
	return CGRectContainsPoint(boundingBox, location);
}

// 正解フレームのアニメーション表示
-(void) blinkFrame
{
	float maxOpacity = 127;
	
	_imgBlinkFrame.opacity = maxOpacity;
	_imgBlinkFrame.visible = YES;
	
	id fadeOut = [CCFadeTo actionWithDuration:0.3 opacity:0];
	id fadeIn = [CCFadeTo actionWithDuration:0.8 opacity:maxOpacity];
	id seq = [CCSequence actions:fadeOut, fadeIn, nil];
	id rep = [CCRepeatForever actionWithAction:seq];
	
	[_imgBlinkFrame runAction:rep];
}

// 通知センタからの通知イベント
-(void) NotifyFromNotificationCenter:(NSNotification*)notification
{
	if (notification.name == TILE_MSG_NOTIFY_TOUCH_HOLD) {
		// いずれかのタイルの長押し通知
		// ガイドナンバーを表示します
		if (_isBlank == NO) {
			// 通常のタイル
			if (_answer == _now) {
				// 正解位置にある場合、正解用の枠を表示します（点滅アクション付き）
				[self blinkFrame];
			}
			else {
				// 不正解位置の場合、ガイドナンバーと不正解用の枠を表示します
				_lblAnswer.visible = YES;
				_imgFrame.visible = YES;
			}
		}
		else {
			// 空白タイルの場合、不正解用の枠を表示します
			_imgFrame.visible = YES;
		}
	}
	else if (notification.name == TILE_MSG_NOTIFY_TOUCH_END) {
		// いずれかのタイルのタッチ終了通知
		// ガイドナンバーと枠の表示を終了します
		_lblAnswer.visible = NO;
		_imgFrame.visible = NO;
		[_imgBlinkFrame stopAllActions];
		_imgBlinkFrame.visible = NO;
	}
	else if (notification.name == TILE_MSG_NOTIFY_TOUCH_MOVE) {
		// いずれかのタイルのタッチ座標移動通知
		if (((Tile*)notification.object).IsTouchHold) {
			// 長押し中である場合、現在のタッチ座標を元にガイドナンバーの表示判定を行ないます
			[self showGuideNumByTouch:[notification.userInfo objectForKey:TILE_MSG_NOTIFY_TOUCH_MOVE]];
		}
	}
	else if (notification.name == TILE_MSG_NOTIFY_SHOW_NUMBER) {
		// いずれかの正解位置のタイルか、あるいは空白タイルのガイドナンバー表示通知
		if ((_answer == _now) && (_isBlank == NO)) {
			// 自タイルが正解タイルである場合、ガイドナンバーを表示します
			_lblAnswer.visible = YES;
		}
		else if (_isBlank && notification.object != self) {
			// 自タイルが空白タイルで且つ通知が自分からで無い場合、ガイドナンバーを消去します
			_lblAnswer.visible = NO;
		}
	}
	else if (notification.name == TILE_MSG_NOTIFY_HIDE_NUMBER) {
		// ガイドナンバー非表示通知
		if (_isBlank || (_answer == _now)) {
			// 空白タイルあるいは正解タイルの場合、ガイドナンバーを消去します
			_lblAnswer.visible = NO;
		}
	}
}

// タッチ地点移動によるガイドナンバー表示
-(void) showGuideNumByTouch:(UITouch*)touch
{
	if ([self containsTouchLocation:touch]) {
		if ((_answer == _now) || _isBlank) {
			// 自タイルが正解値にあるかあるいは空白タイルである場合、ガイドナンバーを表示します
			_lblAnswer.visible = YES;
			// ガイドナンバー表示イベントを流します
			[[NSNotificationCenter defaultCenter] postNotificationName:TILE_MSG_NOTIFY_SHOW_NUMBER object:self];
		}
		else if (_answer != _now) {
			// 不正解タイルの場合、ガイドナンバー非表示イベントを流します
			[[NSNotificationCenter defaultCenter] postNotificationName:TILE_MSG_NOTIFY_HIDE_NUMBER object:self];
		}
	}
}

// スケジュールイベント：画面長押し
-(void) scheduleEventTouchHold:(ccTime)delta
{	
	if (_isTouchBegin == NO) {
		// 自タイルのタッチイベントではない場合、長押し判定は行なわず、処理を抜けます
		return;
	}
	
	// 経過時間を加算します
	_deltaTime += delta;
	
	if (_deltaTime > 2.0) {
		// 経過時間が２秒を超えた場合、長押しとします
		_isTouchHold = YES;
		
		// 長押し通知を発行します
		[[NSNotificationCenter defaultCenter] postNotificationName:TILE_MSG_NOTIFY_TOUCH_HOLD object:self];
		
		// 長押しスケジュールを解除します
		[self unschedule:_cmd];
		
		if ((_answer == _now) || _isBlank) {
			// 正解タイルあるいは空白タイルの場合、ガイドナンバーを表示します
			_lblAnswer.visible = YES;
			
			// ガイドナンバー表示通知を発行します
			[[NSNotificationCenter defaultCenter] postNotificationName:TILE_MSG_NOTIFY_SHOW_NUMBER object:self];
		}
	}
}

@end
