//
//  GameLayer.m
//  15Puzzle
//
//  Created by 久保島 祐磨 on 12/07/19.
//  Copyright 2012年 株式会社 ICT Fractal. All rights reserved.
//

#import "GameLayer.h"
#import "ClearLayer.h"

@interface GameLayer ()

// タイルを並べる
-(void) setTiles;

// タイルをシャッフルする
-(void) shuffle;

// 指定の現在位置を持つタイルオブジェクトを返す
-(Tile*) getTileAtNow:(int)now;

// 通知センタからの通知イベント
-(void) NotifyFromNotificationCenter:(NSNotification*)notification;

// タイルのタップ処理
-(void) tapTile:(Tile*)tile;

@end


@implementation GameLayer

+(CCScene *) scene
{
	CCScene *scene = [CCScene node];
	GameLayer *layer = [GameLayer node];
	[scene addChild: layer];
	
	return scene;
}

-(id) init
{
	if (self = [super init]) {
		// メンバーの初期化
		_tileCount = 25;		// 5x5の24パズルを作ります
		_tileList = nil;
		_actionCount = 0;
		_finishedActionCount = 0;
	}
	
	return self;
}

-(void) dealloc
{
	// メンバーの解放を行ないます
	[_tileList release];
	
	// スーパークラスのdeallocをコールします
	[super dealloc];
}

-(void) onEnter
{
	// スーパークラスのonEnterメソッドをコールします
	[super onEnter];
	
	// 通知センタのオブザーバ登録をします
	[[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(NotifyFromNotificationCenter:) name:nil object:nil];
	
	// 絵をCCTexture2Dオブジェクトに読み込みます
	CCTexture2D* tex = [[[CCTexture2D alloc] initWithCGImage:[UIImage imageNamed:@"image.png"].CGImage resolutionType:kCCResolutioniPhone ] autorelease];
	
	// １辺のタイル枚数を計算します
	int sideTileCount = (int)sqrt((double)_tileCount);
	
	// １枚のタイルサイズを計算します
	CGSize tileSize = CGSizeMake(tex.contentSize.width / sideTileCount, tex.contentSize.height / sideTileCount);
	
	// タイルを管理するため、配列を生成します
	_tileList = [[CCArray alloc] initWithCapacity:_tileCount];
	
	// １６枚のタイルを生成します
	for (int i = 0; i < _tileCount; i++) {
		// 絵から適切な領域を切り抜き、Tileに渡します
		Tile* tile = [Tile spriteWithTexture:tex rect:CGRectMake(tileSize.width * (i % sideTileCount), tileSize.height * (i / sideTileCount), tileSize.width, tileSize.height)];
		
		// 配列に格納します
		[_tileList addObject:tile];
		
		// レイヤーに追加します
		[self addChild:tile z:1];
		
		// タイルの正解位置を記録します
		tile.Answer = i;
		
		// 枠を生成します
		[tile createFrame];
		
		// 最後の１枚であれば、空白タイルの設定を行ないます
		if (i == _tileCount - 1) {
			tile.IsBlank = YES;
		}
	}
	
	// タイルを並べます
	[self setTiles];
	
	// タイルをシャッフルします
	[self shuffle];
}

-(void) onExit
{
	// 通知センタのオブザーバ登録を削除します
	[[NSNotificationCenter defaultCenter] removeObserver:self];
	
	// スーパークラスのonExitをコールします
	[super onExit];
}

// タイルを並べる
-(void) setTiles
{
	CGSize winSize = [[CCDirector sharedDirector] winSize];
	
	// タイルサイズを取得します
	CGSize tileSize = ((Tile*)[_tileList objectAtIndex:0]).contentSize;
	
	// １辺のタイル枚数を計算します
	int sideTileCount = (int)sqrt((double)_tileCount);
	
	// 横の余白を求めます
	float blankX = (winSize.width - tileSize.width * sideTileCount) / 2;
	
	// 縦の余白を求めます
	float blankY = (winSize.height - tileSize.height * sideTileCount) / 2;
	
	// １番のタイルから順に正解位置に配置していきます
	for (int i = 0; i < _tileList.count; i++) {
		Tile* tile = [_tileList objectAtIndex:i];
		tile.position = CGPointMake(blankX + tileSize.width / 2 + tileSize.width * (i % sideTileCount), winSize.height - blankY - tileSize.height / 2 - tileSize.height * (i / sideTileCount));
		
		// タイルの現在位置を記録する
		tile.Now = i;
	}
}

// タイルをシャッフルする
-(void) shuffle
{
	// 空白タイルを取得します
	Tile* blankTile = [_tileList lastObject];
	
	// １辺のタイル枚数を計算します
	int sideTileCount = (int)sqrt((double)_tileCount);
	
	// タイルの総数x100回、空白タイルの移動を行ないます
	for (int i = 0; i < _tileCount * 100; i++) {
		// チェック用の変数を無効な数値で初期化します
		int checkPosition = -1;
		
		// 空白タイルの移動方向をランダムに決定します
		int direction  = arc4random() % 4;
		switch (direction) {
			case 0:
				// 上に移動した場合の位置番号を設定します
				checkPosition = blankTile.Now - sideTileCount;
				if (checkPosition > 0) {
				}
				else {
					// これ以上上に移動できない場合、値をリセットします
					checkPosition = -1;
				}
				break;
				
			case 1:
				// 右に移動した場合の位置番号を設定します
				checkPosition = blankTile.Now + 1;
				if (checkPosition % sideTileCount != 0) {
				}
				else {
					// これ以上右に移動できない場合、値をリセットします
					checkPosition = -1;
				}
				break;
				
			case 2:
				// 下に移動した場合の位置番号を設定します
				checkPosition = blankTile.Now + sideTileCount;
				if (checkPosition < _tileList.count) {
				}
				else {
					// これ以上下に移動できない場合、値をリセットします
					checkPosition = -1;
				}
				break;
				
			case 3:
				// 左に移動した場合の位置番号を設定します
				checkPosition = blankTile.Now - 1;
				if (checkPosition % sideTileCount != sideTileCount - 1) {
				}
				else {
					// これ以上左に移動できない場合、値をリセットします
					checkPosition = -1;
				}
				break;
				
			default:
				checkPosition = -1;
				break;
		}
		
		if (checkPosition > -1) {
			// 移動先のタイルを取得します
			Tile* tile = [self getTileAtNow:checkPosition];
			
			// 移動先タイルと空白タイルの位置情報を入れ替えます
			int		tempIndex = blankTile.Now;
			CGPoint	tempPosition = blankTile.position;
			blankTile.position = tile.position;
			blankTile.Now = tile.Now;
			tile.Now = tempIndex;
			tile.position = tempPosition;
		}
	}
}

// 指定の現在位置を持つタイルオブジェクトを返す
-(Tile*) getTileAtNow:(int)now
{
	Tile* result = nil;
	
	for (Tile* tile in _tileList) {
		if (tile.Now == now) {
			result = tile;
			break;
		}
	}
	
	return result;
}

// 通知センタからの通知イベント
-(void) NotifyFromNotificationCenter:(NSNotification*)notification
{	
	if (notification.name == TILE_MSG_NOTIFY_TAP) {
		// タイルのタップイベント
		// タイルのタップ処理をコールします
		[self tapTile:notification.object];
	}
}

// タイルのタップ
-(void) tapTile:(Tile*)tile
{
	// タイルの検索結果
	BOOL checkResult = NO;
	
	// １辺のタイル枚数を計算します
	int sideTileCount = (int)sqrt((double)_tileCount);
	
	// チェックに使用する一時的な配列を生成します
	CCArray* searchList = [CCArray arrayWithCapacity:sideTileCount];
	
	// 空白タイルを取得します
	Tile* blankTile = [_tileList lastObject];
	
	// 空白タイルを基準として、４方向一列をチェックします
	// 上
	int checkPosition = blankTile.Now - sideTileCount;
	while (checkPosition > -1) {
		[searchList addObject:[self getTileAtNow:checkPosition]];
		if (tile.Now == checkPosition) {
			checkResult = YES;
			break;
		}
		checkPosition -= sideTileCount;
	}
	
	// 右（上方向にタップしたタイルが見つかった場合、チェックしません）
	if (checkResult == NO) {
		[searchList removeAllObjects];
		checkPosition = blankTile.Now + 1;
		while ((checkPosition % sideTileCount != 0) && (checkPosition < _tileList.count)) {
			[searchList addObject:[self getTileAtNow:checkPosition]];
			if (tile.Now == checkPosition) {
				checkResult = YES;
				break;
			}
			checkPosition++;
		}
	}
	
	// 下（上あるいは右方向にタップしたタイルが見つかった場合、チェックしません）
	if (checkResult == NO) {
		[searchList removeAllObjects];
		checkPosition = blankTile.Now + sideTileCount;
		while (checkPosition < _tileList.count) {
			[searchList addObject:[self getTileAtNow:checkPosition]];
			if (tile.Now == checkPosition) {
				checkResult = YES;
				break;
			}
			checkPosition += sideTileCount;
		}
	}
	
	// 左（上、右、下何れかの方行にタップしたタイルが見つかった場合、チェックしません）
	if (checkResult == NO) {
		[searchList removeAllObjects];
		checkPosition = blankTile.Now - 1;
		while ((checkPosition % sideTileCount != sideTileCount - 1) && (checkPosition > -1)) {
			[searchList addObject:[self getTileAtNow:checkPosition]];
			if (tile.Now == checkPosition) {
				checkResult = YES;
				break;
			}
			checkPosition--;
		}
	}
	
	// checkResultがYESの場合、searchListに格納されているTileオブジェクトが移動するタイルとなります
	if (checkResult) {
		// 各タイルに移動アニメーションを適用するため、CCActionを保持する為の配列を生成します
		CCArray* actionList = [CCArray arrayWithCapacity:searchList.count];
		
		// アクションカウントを初期化します
		_finishedActionCount = 0;
		
		// searchListに含まれるTileオブジェクトの数分、移動アニメーションを生成します
		for (Tile* tile in searchList) {
			// 空白タイルのNow値を一時変数に保持します
			int tempIndex = blankTile.Now;
			
			// 空白タイルの座標へ移動するアクションを生成します
			id move = [CCMoveTo actionWithDuration:0.1 position:blankTile.position];
			
			// クリア判定を行なう為のアクションを生成します
			id moveEnd = [CCCallBlock actionWithBlock:^{
				// 完了アクション数を+1します
				_finishedActionCount++;
				
				if (_finishedActionCount == _actionCount) {
					// すべてのアクションが完了した場合、クリア判定を行ないます
					BOOL isClear = YES;
					for (Tile* tile in _tileList) {
						// すべてのタイルが正解位置にあれば、クリアとします
						if (tile.Answer != tile.Now) {
							isClear = NO;
							break;
						}
					}
					
					if (isClear) {
						// ClearLayerを表示します
						[self addChild:[ClearLayer node] z:2];
						
						// 空白タイルのIsBlankプロパティをNOに設定し、完成絵を表示します
						((Tile*)[_tileList lastObject]).IsBlank = NO;
					}
				}
			}];
			
			// 移動アクションと移動後処理アクションをシーケンスにします
			id seq = [CCSequence actions:move, moveEnd, nil];
			
			// 空白タイルと移動タイルの位置情報を交換します（移動タイルのposition以外）
			blankTile.position = tile.position;
			blankTile.Now = tile.Now;
			tile.Now = tempIndex;
			
			// 配列にアクションを格納します
			[actionList addObject:seq];
		}
		
		// アクションの総数を保持します
		_actionCount = actionList.count;
		
		// 各移動アクションオブジェクトを、タイルに適用します
		for (int i = 0; i < searchList.count; i++) {
			Tile* tile = [searchList objectAtIndex:i];
			id action = [actionList objectAtIndex:i];
			[tile runAction:action];
		}
	}
}

@end
